from mongoengine import Document, StringField, DateTimeField, BooleanField
from datetime import datetime, timedelta
import uuid
import secrets


class ResetToken(Document):
    """Password Reset Token model"""
    meta = {'collection': 'reset_tokens'}

    token_id = StringField(required=True, unique=True, default=lambda: str(uuid.uuid4()))
    user_id = StringField(required=True)
    email = StringField(required=True)
    reset_token = StringField(required=True, unique=True)
    is_used = BooleanField(default=False)
    is_expired = BooleanField(default=False)
    created_at = DateTimeField(default=datetime.utcnow)
    expires_at = DateTimeField(required=True)
    used_at = DateTimeField()

    # Device and location info
    device_type = StringField()
    device_fingerprint = StringField()
    mac_id = StringField()
    ip_address = StringField()
    location = StringField()

    # Related OTP
    otp_id = StringField()

    def is_valid(self):
        """Check if reset token is valid"""
        if self.is_used:
            return False
        if self.is_expired:
            return False
        if datetime.utcnow() > self.expires_at:
            self.is_expired = True
            self.save()
            return False
        return True

    def mark_as_used(self):
        """Mark token as used"""
        self.is_used = True
        self.used_at = datetime.utcnow()
        self.save()

    @staticmethod
    def generate_reset_token():
        """Generate secure random reset token"""
        return secrets.token_urlsafe(32)

    def __repr__(self):
        return f'<ResetToken {self.email}>'
